<?php /** @noinspection PhpUndefinedClassInspection */
/**
 * Pricer for Elementor
 * Pricer is a Elementor widget for creating attractive price tables and price cards
 * Exclusively on https://1.envato.market/pricer-elementor
 *
 * @encoding        UTF-8
 * @version         2.0.0
 * @copyright       (C) 2018 - 2021 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Cherviakov Vlad (vladchervjakov@gmail.com), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\PricerElementor;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

use Exception;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Repeater;
use Elementor\Core\Schemes\Typography;
use Elementor\Core\Schemes\Color;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Merkulove\PricerElementor\Unity\Plugin as UnityPlugin;
use Merkulove\PricerElementor\Unity\ElementorControls;

/** @noinspection PhpUnused */
/**
 * Pricer - Custom Elementor Widget.
 **/
class pricer_elementor extends Widget_Base {

    /**
     * Use this to sort widgets.
     * A smaller value means earlier initialization of the widget.
     * Can take negative values.
     * Default widgets and widgets from 3rd party developers have 0 $mdp_order
     **/
    public $mdp_order = 1;

    /**
     * Widget base constructor.
     * Initializing the widget base class.
     *
     * @access public
     * @throws Exception If arguments are missing when initializing a full widget instance.
     * @param array      $data Widget data. Default is an empty array.
     * @param array|null $args Optional. Widget default arguments. Default is null.
     *
     * @return void
     **/
    public function __construct( $data = [], $args = null ) {

        parent::__construct( $data, $args );

        wp_register_style( 'mdp-pricer-elementor-admin', UnityPlugin::get_url() . 'src/Merkulove/Unity/assets/css/elementor-admin' . UnityPlugin::get_suffix() . '.css', [], UnityPlugin::get_version() );
        wp_register_style( 'mdp-pricer-elementor', UnityPlugin::get_url() . 'css/pricer-elementor' . UnityPlugin::get_suffix() . '.css', [], UnityPlugin::get_version() );
        wp_register_style( 'animate-css', UnityPlugin::get_url() . 'css/hover' . UnityPlugin::get_suffix() . '.css', [], UnityPlugin::get_version() );


    }

    /**
     * Return a widget name.
     *
     * @return string
     **/
    public function get_name() {

        return 'mdp-pricer-elementor';

    }

    /**
     * Return the widget title that will be displayed as the widget label.
     *
     * @return string
     **/
    public function get_title() {

        return esc_html__( 'Pricer', 'pricer-elementor' );

    }

    /**
     * Set the widget icon.
     *
     * @return string
     */
    public function get_icon() {

        return 'mdp-pricer-elementor-widget-icon';

    }

    /**
     * Set the category of the widget.
     *
     * @return array with category names
     **/
    public function get_categories() {

        return [ 'general' ];

    }

    /**
     * Get widget keywords. Retrieve the list of keywords the widget belongs to.
     *
     * @access public
     *
     * @return array Widget keywords.
     **/
    public function get_keywords() {

        return [ 'Merkulove', 'Pricer', 'Pricing', 'Cost', 'Tariff', 'Rate' ];

    }

    /**
     * Get style dependencies.
     * Retrieve the list of style dependencies the widget requires.
     *
     * @access public
     *
     * @return array Widget styles dependencies.
     **/
    public function get_style_depends() {

        return [ 'mdp-pricer-elementor', 'animate-css', 'mdp-pricer-elementor-admin' ];

    }

	/**
	 * Get script dependencies.
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @access public
     *
	 * @return array Element scripts dependencies.
	 **/
	public function get_script_depends() {

		return [];

    }

    /**
     * Controls in the Content Tab – Layout Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function content_layout() {

        /** Layout tab. */
        $this->start_controls_section( 'section_layout', [ 'label' => esc_html__( 'Layout', 'pricer-elementor' ) ] );

        /** Repeater Horizontal layout. */
        $repeater = new Repeater();

        /** Layout. */
        $repeater->add_control(
            'plan_layout_select',
            [
                'label' => esc_html__( 'Layout', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'header',
                'options' => [
                    'header'  => esc_html__( 'Header', 'pricer-elementor' ),
                    'price' => esc_html__( 'Price', 'pricer-elementor' ),
                    'period' => esc_html__( 'Period', 'pricer-elementor' ),
                    'divider' => esc_html__( 'Divider', 'pricer-elementor' ),
                    'list' => esc_html__( 'Features', 'pricer-elementor' ),
                    'description' => esc_html__( 'Description', 'pricer-elementor' ),
                    'link' => esc_html__( 'Button', 'pricer-elementor' ),
                ],
            ]
        );

        /** Layout list. */
        $this->add_control(
            'plan_layout',
            [
                'label' => esc_html__( 'Layout list', 'pricer-elementor' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    ['plan_layout_select' => 'header' ],
                    ['plan_layout_select' => 'price' ],
                    ['plan_layout_select' => 'period' ],
                    ['plan_layout_select' => 'list' ],
                    ['plan_layout_select' => 'description' ],
                    ['plan_layout_select' => 'link' ],
                ],
                'title_field' => 'Layout  - {{{plan_layout_select}}}',
            ]
        );

        /** End section content. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Content Tab – Header Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function content_header() {

        /** Header tab. */
        $this->start_controls_section( 'section_header', [ 'label' => esc_html__( 'Header', 'pricer-elementor' ), ] );

        /** Title. */
        $this->add_control(
            'plan_header',
            [
                'label' => esc_html__( 'Title', 'pricer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Title', 'pricer-elementor' ),
                'placeholder' => esc_html__( 'Title', 'pricer-elementor' ),
            ]
        );

        /** HTML Tag. */
        $this->add_control(
            'plan_tag',
            [
                'label' => esc_html__( 'HTML Tag', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'h1' => 'H1',
                    'h2' => 'H2',
                    'h3' => 'H3',
                    'h4' => 'H4',
                    'h5' => 'H5',
                    'h6' => 'H6',
                    'div' => 'div',
                    'span' => 'span',
                    'p' => 'p',
                ],
                'default' => 'h4',
            ]
        );

        /** Alignment. */
        $this->add_responsive_control(
            'header_align',
            [
                'label' => esc_html__( 'Alignment', 'pricer-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-heading' => 'text-align: {{header_align}};',
                ],
                'toggle' => true,
            ]
        );

        /** Show Subheader. */
        $this->add_control(
            'show_sub_title',
            [
                'label' => esc_html__( 'Show Subheader', 'pricer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'pricer-elementor' ),
                'label_off' => esc_html__( 'Hide', 'pricer-elementor' ),
                'return_value' => 'yes',
                'separator' => 'before'
            ]
        );

        /** Subheader Position. */
        $this->add_control(
            'sub_title_position',
            [
                'label' => esc_html__( 'Subheader Position', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'top' => 'Top',
                    'bottom' => 'Bottom',
                ],
                'default' => 'bottom',
                'condition'   => ['show_sub_title' => 'yes']
            ]
        );

        /** Subheader. */
        $this->add_control(
            'plan_sub_name',
            [
                'label' => esc_html__( 'Subheader', 'pricer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Subheader', 'pricer-elementor' ),
                'placeholder' => esc_html__( 'Subheader', 'pricer-elementor' ),
                'condition'   => ['show_sub_title' => 'yes']
            ]
        );

        /** End section style content. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Content Tab – Price Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function content_price() {

        /** Price tab. */
        $this->start_controls_section( 'section_price', [ 'label' => esc_html__( 'Price', 'pricer-elementor' ) ] );

        /** Price. */
        $this->add_control(
            'plan_price',
            [
                'label' => esc_html__( 'Price', 'pricer-elementor' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 0,
                'step' => .01,
                'default' => 9.99
            ]
        );

        /** Decimal Position. */
        $this->add_control(
            'decimal_position',
            [
                'label' => esc_html__( 'Decimal Position', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'default' => 'Default',
                    'top' => 'Top',
                    'bottom' => 'Bottom',
                ],
                'default' => 'default',
            ]
        );

        /** Point decimal.  */
        $this->add_control(
            'decimal_point',
            [
                'label' => esc_html__( 'Point decimal', 'pricer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( '.', 'pricer-elementor' ),
                'placeholder' => esc_html__( 'Point decimal', 'pricer-elementor' ),
                'condition' => ['decimal_position' => 'default']
            ]
        );

        /** Currency Position. */
        $this->add_control(
            'currency_position',
            [
                'label' => esc_html__( 'Currency Position', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'left' => 'Left',
                    'left-top' => 'Left Top',
                    'left-bottom' => 'Left Bottom',
                    'right' => 'Right',
                    'right-top' => 'Right Top',
                    'right-bottom' => 'Right Bottom',
                ],
                'default' => 'left',
            ]
        );

        /** Currency symbol. */
        $this->add_control(
            'plan_currency',
            [
                'label' => esc_html__( 'Currency symbol', 'pricer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( '$', 'pricer-elementor' ),
                'placeholder' => esc_html__( '$', 'pricer-elementor' ),
            ]
        );

        /** Alignment. */
        $this->add_responsive_control(
            'price_align',
            [
                'label' => esc_html__( 'Alignment', 'pricer-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-price' => 'text-align: {{header_align}};',
                ],
                'toggle' => true,
            ]
        );

        /** Show zeros in the remainder. */
        $this->add_control(
            'show_zeros_original_price',
            [
                'label' => esc_html__( 'Show tenths', 'pricer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'pricer-elementor' ),
                'label_off' => esc_html__( 'Hide', 'pricer-elementor' ),
                'return_value' => 'yes',
            ]
        );

        /** Show Previous Price. */
        $this->add_control(
            'show_original_price',
            [
                'label' => esc_html__( 'Show Previous Price', 'pricer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'pricer-elementor' ),
                'label_off' => esc_html__( 'Hide', 'pricer-elementor' ),
                'return_value' => 'yes',
                'separator' => 'before',
            ]
        );

        /** Previous Price. */
        $this->add_control(
            'plan_original_price',
            [
                'label' => esc_html__( 'Previous Price', 'pricer-elementor' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 0,
                'step' => .01,
                'default' => 15,
                'condition'   => ['show_original_price' => 'yes']
            ]
        );

        /** Previous Price position */
        $this->add_control(
            'original_price_position',
            [
                'label' => esc_html__( 'Previous Price Position', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'left' => 'Left',
                    'right' => 'Right',
                ],
                'default' => 'left',
                'condition'   => ['show_original_price' => 'yes']
            ]
        );

        /** End section style content. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Content Tab – Period Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function content_period() {

        /** Style price tab. */
        $this->start_controls_section( 'content_period', [ 'label' => esc_html__( 'Period', 'pricer-elementor' ) ] );

        /** Period. */
        $this->add_control(
            'plan_period',
            [
                'label' => esc_html__( 'Period', 'pricer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Period', 'pricer-elementor' ),
                'placeholder' => esc_html__( 'Period', 'pricer-elementor' ),
            ]
        );

        /** HTML Tag. */
        $this->add_control(
            'period_tag',
            [
                'label' => esc_html__( 'HTML Tag', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'h1' => 'H1',
                    'h2' => 'H2',
                    'h3' => 'H3',
                    'h4' => 'H4',
                    'h5' => 'H5',
                    'h6' => 'H6',
                    'div' => 'div',
                    'span' => 'span',
                    'p' => 'p',
                ],
                'default' => 'p',
            ]
        );

        /** Alignment. */
        $this->add_responsive_control(
            'period_align',
            [
                'label' => esc_html__( 'Alignment', 'pricer-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-period' => 'text-align: {{header_align}};',
                ],
                'toggle' => true,
            ]
        );

        /** End section style price. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Content Tab – Features Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function content_features() {

        /** Features tab. */
        $this->start_controls_section( 'section_features', [ 'label' => esc_html__( 'Features', 'pricer-elementor' ) ] );

        /** Alignment. */
        $this->add_responsive_control(
            'style_align_active_features',
            [
                'label' => esc_html__( 'Alignment', 'pricer-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-list' => 'text-align: {{header_align}};',
                ],
                'toggle' => true,
            ]
        );

        $repeater = new Repeater();

        /** Description text. */
        $repeater->add_control(
            'description_text',
            [
                'label' => esc_html__( 'Description text', 'pricer-elementor' ),
                'type'  => Controls_Manager::TEXT,
            ]
        );

        /** Plan included. */
        $repeater->add_control(
            'plan_included',
            [
                'label' => esc_html__( 'Plan included', 'pricer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'pricer-elementor' ),
                'label_off' => esc_html__( 'No', 'pricer-elementor' ),
                'return_value' => 'yes',
            ]
        );

        /** Plan list. */
        $this->add_control(
            'plan_list',
            [
                'label' => esc_html__( 'Plan list', 'pricer-elementor' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    ['description_text' => 'Personal Locker', 'plan_included' => 'yes'],
                    ['description_text' => 'Free access 24/7', 'plan_included' => ''],
                    ['description_text' => 'Personal Trainer', 'plan_included' => 'yes'],
                    ['description_text' => 'Nutrition plan', 'plan_included' => ''],
                    ['description_text' => 'Free massage', 'plan_included' => 'yes']
                ],
                'title_field' => 'Plan list  - {{{description_text}}}',
            ]
        );

        /** End section style features. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Content Tab – Description Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function content_description() {

        /** Description tab. */
        $this->start_controls_section( 'section_description', [ 'label' => esc_html__( 'Description', 'pricer-elementor' ) ] );

        /** Description. */
        $this->add_control(
            'plan_description',
            [
                'label' => esc_html__( 'Description', 'pricer-elementor' ),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 10,
                'default' => esc_html__( 'Description', 'pricer-elementor' ),
                'placeholder' => esc_html__( 'Description', 'pricer-elementor' ),
            ]
        );

        /** HTML Tag. */
        $this->add_control(
            'description_tag',
            [
                'label' => esc_html__( 'HTML Tag', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'h1' => 'H1',
                    'h2' => 'H2',
                    'h3' => 'H3',
                    'h4' => 'H4',
                    'h5' => 'H5',
                    'h6' => 'H6',
                    'div' => 'div',
                    'span' => 'span',
                    'p' => 'p',
                ],
                'default' => 'p',
            ]
        );

        /** Alignment. */
        $this->add_responsive_control(
            'style_align_description',
            [
                'label' => esc_html__( 'Alignment', 'pricer-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                    'justify' => [
                        'title' => esc_html__( 'Justify', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-justify',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-description' => 'text-align: {{header_align}};',
                ],
                'toggle' => true,
            ]
        );

        /** End section style description. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Content Tab – Button Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function content_button() {

        /** Button tab. */
        $this->start_controls_section( 'section_button', [ 'label' => esc_html__( 'Button', 'pricer-elementor' ) ] );

        /** Link text. */
        $this->add_control(
            'link_text',
            [
                'label' => esc_html__( 'Link text', 'pricer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Link text', 'pricer-elementor' ),
                'placeholder' => esc_html__( 'Link text', 'pricer-elementor' ),
            ]
        );

        /** Alignment. */
        $this->add_responsive_control(
            'style_align_button',
            [
                'label' => esc_html__( 'Alignment', 'pricer-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link' => 'text-align: {{header_align}};',
                ],
                'toggle' => true,
            ]
        );

        /** Plan link. */
        $this->add_control(
            'plan_link',
            [
                'label' => esc_html__( 'Plan link', 'pricer-elementor' ),
                'type' => Controls_Manager::URL,
                'placeholder' => esc_html__( 'Plan link', 'pricer-elementor' ),
                'show_external' => true,
                'default' => [
                    'url' => '',
                    'is_external' => true,
                    'nofollow' => true,
                ],
            ]
        );

        /** End section style content. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Style Tab – Header Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function style_header() {

        /** Style header tab. */
        $this->start_controls_section( 'style_header',
            [
                'label' => esc_html__( 'Header', 'pricer-elementor' ), 'tab' => Controls_Manager::TAB_STYLE,
            ] );

        /** Header width. */
        $this->add_responsive_control(
            'header_width',
            [
                'label' => esc_html__( 'Width', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ '%', 'px', 'vh' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    'vh' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'desktop_default' => [
                    'size' => 100,
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-heading' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        /** Margin. */
        $this->add_responsive_control(
            'header_margin',
            [
                'label' => esc_html__( 'Margin', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-heading' => 'margin: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Padding. */
        $this->add_responsive_control(
            'header_padding',
            [
                'label' => esc_html__( 'Padding', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-header' => 'padding: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        $this->start_controls_tabs('tabs_header_style');

        /** Text style tab. */
        $this->start_controls_tab(
            'tab_style_text',
            [
                'label' => esc_html__('Text style', 'pricer-elementor')
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'color_header',
            [
                'label' => esc_html__( 'Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-header' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'header_typography',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-pricer-header',
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'header_shadow',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-pricer-header',
            ]
        );

        $this->end_controls_tab();

        /** Box tab. */
        $this->start_controls_tab(
            'tab_style_box',
            [
                'label' => esc_html__('Box style', 'pricer-elementor')
            ]
        );

        /** Background. */
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'header_background',
                'label' => esc_html__( 'Background', 'pricer-elementor' ),
                'types' => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .mdp-pricer-header',
            ]
        );

        /** Inner Background. */
        $this->add_control(
            'header_background_effect',
            [
                'label' => esc_html__( 'Inner Background', 'pricer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'pricer-elementor' ),
                'label_off' => esc_html__( 'Hide', 'pricer-elementor' ),
                'return_value' => 'yes',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        /** End section style header. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Style Tab – Subheader Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function style_subheader() {

        /** Style subheader tab. */
        $this->start_controls_section( 'style_subheader',
            [
                'label' => esc_html__( 'Subheader', 'pricer-elementor' ), 'tab' => Controls_Manager::TAB_STYLE,
            ] );

        /** Padding. */
        $this->add_responsive_control(
            'subheader_padding',
            [
                'label' => esc_html__( 'Padding', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-subheader' => 'padding: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        $this->start_controls_tabs('tabs_subtitle_style');

        /** Text style tab. */
        $this->start_controls_tab(
            'tab_style_subtext',
            [
                'label' => esc_html__('Text style', 'pricer-elementor')
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'color_subheader',
            [
                'label' => esc_html__( 'Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-subheader' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'subheader_typography',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-pricer-subheader',
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'subheader_shadow',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-pricer-subheader',
            ]
        );

        $this->end_controls_tab();

        /** Box tab. */
        $this->start_controls_tab(
            'tab_style_sub_box',
            [
                'label' => esc_html__('Box style', 'pricer-elementor')
            ]
        );

        /** Background. */
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'subheader_background',
                'label' => esc_html__( 'Background', 'pricer-elementor' ),
                'types' => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .mdp-pricer-subheader',
            ]
        );

        /** Inner Background. */
        $this->add_control(
            'subheader_background_effect',
            [
                'label' => esc_html__( 'Inner Background', 'pricer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'pricer-elementor' ),
                'label_off' => esc_html__( 'Hide', 'pricer-elementor' ),
                'return_value' => 'yes',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        /** End section style subheader. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Style Tab – Price Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function style_price() {

        /** Style price tab. */
        $this->start_controls_section( 'style_price',
            [
                'label' => esc_html__( 'Price', 'pricer-elementor' ), 'tab' => Controls_Manager::TAB_STYLE,
            ] );

        /** Price width. */
        $this->add_responsive_control(
            'price_width',
            [
                'label' => esc_html__( 'Width', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'size_units' => [ '%', 'px', 'vh' ],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    'vh' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'desktop_default' => [
                    'size' => 100,
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-price' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        /** Margin. */
        $this->add_responsive_control(
            'price_margin',
            [
                'label' => esc_html__( 'Margin', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-price' => 'margin: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Padding. */
        $this->add_responsive_control(
            'price_padding',
            [
                'label' => esc_html__( 'Padding', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-price' => 'padding: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Background */
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'price_background',
                'label' => esc_html__( 'Background', 'pricer-elementor' ),
                'types' => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .mdp-pricer-price',
            ]
        );

        /** Price. */
        $this->add_control(
            'style_header_price',
            [
                'label' => esc_html__( 'Price', 'pricer-elementor' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        /** Space left. */
        $this->add_responsive_control(
            'price_space_left',
            [
                'label' => esc_html__( 'Space left', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-style' => 'padding-left: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        /** Space right. */
        $this->add_responsive_control(
            'price_space_right',
            [
                'label' => esc_html__( 'Space right', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-style' => 'padding-right: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'color_price',
            [
                'label' => esc_html__( 'Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-style' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'price_typography',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-pricer-style',
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'price_shadow',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-pricer-style',
            ]
        );

        /** Price currency symbol. */
        $this->add_control(
            'style_header_currency_symbol',
            [
                'label' => esc_html__( 'Price currency symbol', 'pricer-elementor' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'color_currency_symbol',
            [
                'label' => esc_html__( 'Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-currency' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'currency_symbol_typography',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-pricer-currency',
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'currency_symbol_shadow',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-pricer-currency',
            ]
        );

        /** Decimal Price. */
        $this->add_control(
            'style_header_decimal',
            [
                'label' => esc_html__( 'Decimal Price', 'pricer-elementor' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'color_decimal',
            [
                'label' => esc_html__( 'Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-decimal' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'decimal_typography',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-pricer-decimal',
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'decimal_shadow',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-pricer-decimal',
            ]
        );

        /** Previous Price. */
        $this->add_control(
            'style_header_original_price',
            [
                'label' => esc_html__( 'Previous Price', 'pricer-elementor' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [ 'show_original_price' => 'yes' ]
            ]
        );

        /** Space left. */
        $this->add_responsive_control(
            'original_price_space_left',
            [
                'label' => esc_html__( 'Space left', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-original-pricer-style' => 'padding-left: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [ 'show_original_price' => 'yes' ]
            ]
        );

        /** Space right. */
        $this->add_responsive_control(
            'original_price_space_right',
            [
                'label' => esc_html__( 'Space right', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-original-pricer-style' => 'padding-right: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [ 'show_original_price' => 'yes' ]
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'color_original_price',
            [
                'label' => esc_html__( 'Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-original-pricer-style' => 'color: {{VALUE}}',
                ],
                'condition' => [ 'show_original_price' => 'yes' ]
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'original_price_typography',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-original-pricer-style',
                'condition' => [ 'show_original_price' => 'yes' ]
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'original_price_shadow',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-original-pricer-style',
                'condition' => [ 'show_original_price' => 'yes' ]
            ]
        );

        /** Previous Price currency symbol. */
        $this->add_control(
            'style_header_original_currency_symbol',
            [
                'label' => esc_html__( 'Previous Price currency symbol', 'pricer-elementor' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [ 'show_original_price' => 'yes' ]
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'color_original_currency_symbol',
            [
                'label' => esc_html__( 'Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-original-pricer-currency' => 'color: {{VALUE}}',
                ],
                'condition' => [ 'show_original_price' => 'yes' ]
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'currency_original_symbol_typography',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-original-pricer-currency',
                'condition' => [ 'show_original_price' => 'yes' ]
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'original_currency_symbol_shadow',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-original-pricer-currency',
                'condition' => [ 'show_original_price' => 'yes' ]
            ]
        );

        /** End section style price. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Style Tab – Period Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function style_period() {

        /** Style price tab. */
        $this->start_controls_section( 'style_period',
            [
                'label' => esc_html__( 'Period', 'pricer-elementor' ), 'tab' => Controls_Manager::TAB_STYLE,
            ] );

        /** Period width. */
        $this->add_responsive_control(
            'period_width',
            [
                'label' => esc_html__( 'Width', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'size_units' => [ '%', 'px', 'vh' ],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    'vh' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'desktop_default' => [
                    'size' => 100,
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-period' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        /** Margin. */
        $this->add_responsive_control(
            'period_margin',
            [
                'label' => esc_html__( 'Margin', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-period' => 'margin: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Padding. */
        $this->add_responsive_control(
            'period_padding',
            [
                'label' => esc_html__( 'Padding', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-period' => 'padding: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'style_color_period',
            [
                'label' => esc_html__( 'Ink Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-period' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'style_typography_period',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-pricer-period',
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'style_shadow_period',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-pricer-period',
            ]
        );

        /** Background */
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'period_background',
                'label' => esc_html__( 'Background', 'pricer-elementor' ),
                'types' => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .mdp-pricer-period',
            ]
        );

        /** End section style price. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Style Tab – Divider Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function style_divider() {

        /** Style divider tab. */
        $this->start_controls_section( 'style_divider',
            [
                'label' => esc_html__( 'Divider', 'pricer-elementor' ), 'tab' => Controls_Manager::TAB_STYLE,
            ] );

        /** Divider width. */
        $this->add_responsive_control(
            'divider_width',
            [
                'label' => esc_html__( 'Width', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'size_units' => [ '%', 'px', 'vh' ],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    'vh' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'desktop_default' => [
                    'size' => 100,
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-divider' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        /** Height. */
        $this->add_responsive_control(
            'divider_height',
            [
                'label' => esc_html__( 'Height', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'default' => [
                    'unit' => 'px',
                    'size' => 1,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-divider' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        /** Alignment. */
        $this->add_responsive_control(
            'divider_align',
            [
                'label' => esc_html__( 'Alignment', 'pricer-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'pricer-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-box-divider' => 'text-align: {{header_align}};',
                ],
                'toggle' => true,
            ]
        );

        /** Margin. */
        $this->add_responsive_control(
            'divider_margin',
            [
                'label' => esc_html__( 'Margin', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-box-divider' => 'margin: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Padding. */
        $this->add_responsive_control(
            'divider_padding',
            [
                'label' => esc_html__( 'Padding', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-box-divider' => 'padding: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Line color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'border_color_divider',
            [
                'label' => esc_html__( 'Line color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_1,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-divider' => 'border-top: {{divider_height.size}}{{divider_height.unit}} {{divider_border_type.VALUE}} {{VALUE}};',
                ],
            ]
        );

        /** Line type. */
        $this->add_control(
            'divider_border_type',
            [
                'label' => esc_html__( 'Line type', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'solid',
                'options' => [
                    'double' => esc_html__( 'Double', 'pricer-elementor' ),
                    'dashed' => esc_html__( 'Dashed', 'pricer-elementor' ),
                    'dotted' => esc_html__( 'Dotted', 'pricer-elementor' ),
                    'solid' => esc_html__( 'Solid', 'pricer-elementor' ),
                ],
                'separator' => 'before',
            ]
        );

        /** Line radius. */
        $this->add_responsive_control(
            'divider_border_radius',
            [
                'label' => esc_html__( 'Line radius', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-divider' => 'border-radius: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
            ]
        );

        /** Background */
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'divider_background',
                'label' => esc_html__( 'Background', 'pricer-elementor' ),
                'types' => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .mdp-pricer-box-divider',
            ]
        );

        /** End section style divider. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Style Tab – Features Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function style_features() {

        /** Style features tab. */
        $this->start_controls_section( 'style_features',
            [
                'label' => esc_html__( 'Features', 'pricer-elementor' ), 'tab' => Controls_Manager::TAB_STYLE,
            ] );

        /** Features width. */
        $this->add_responsive_control(
            'features_width',
            [
                'label' => esc_html__( 'Width', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'size_units' => [ '%', 'px', 'vh' ],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    'vh' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'desktop_default' => [
                    'size' => 100,
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-list' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        /** Margin. */
        $this->add_responsive_control(
            'features_margin',
            [
                'label' => esc_html__( 'Margin', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-list' => 'margin: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Padding. */
        $this->add_responsive_control(
            'features_padding',
            [
                'label' => esc_html__( 'Padding', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-list' => 'padding: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** List item padding. */
        $this->add_responsive_control(
            'features_space',
            [
                'label' => esc_html__( 'Space Between', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-list li' => 'padding-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        /** Active Features. */
        $this->add_control(
            'style_header_active_features',
            [
                'label' => esc_html__( 'Active', 'pricer-elementor' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'style_color_active_features',
            [
                'label' => esc_html__( 'Text Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-active-features' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'style_typography_active_features',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-pricer-active-features',
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'style_shadow_active_features',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-pricer-active-features',
            ]
        );

        /** Icon active. */
        $this->add_control(
            'icon_active',
            [
                'label' => esc_html__( 'Icon', 'pricer-elementor' ),
                'type' => Controls_Manager::ICONS,
                'default' => [
                    'value' => 'far fa-check-circle',
                    'library' => 'fa-regular',
                ],
            ]
        );

        /** Icon Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'style_color_active_icon_features',
            [
                'label' => esc_html__( 'Icon Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-active-icon-color' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Inactive Features. */
        $this->add_control(
            'style_header_inactive_features',
            [
                'label' => esc_html__( 'Inactive', 'pricer-elementor' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'style_color_inactive_features',
            [
                'label' => esc_html__( 'Text Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-inactive-features' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'style_typography_inactive_features',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-pricer-inactive-features',
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'style_shadow_inactive_features',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-pricer-inactive-features',
            ]
        );

        /** Icon inactive. */
        $this->add_control(
            'icon_inactive',
            [
                'label' => esc_html__( 'Icon', 'pricer-elementor' ),
                'type' => Controls_Manager::ICONS,
                'default' => [
                    'value' => 'far fa-times-circle',
                    'library' => 'fa-regular',
                ],
            ]
        );

        /** Icon Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'style_color_inactive_icon_features',
            [
                'label' => esc_html__( 'Icon Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-inactive-icon-color' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** End section style features. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Style Tab – Description Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function style_description() {

        /** Description tab. */
        $this->start_controls_section( 'style_description',
            [
                'label' => esc_html__( 'Description', 'pricer-elementor' ), 'tab' => Controls_Manager::TAB_STYLE,
            ] );

        /** Description width. */
        $this->add_responsive_control(
            'description_width',
            [
                'label' => esc_html__( 'Width', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'size_units' => [ '%', 'px', 'vh' ],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    'vh' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'desktop_default' => [
                    'size' => 100,
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-description' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        /** Margin. */
        $this->add_responsive_control(
            'description_margin',
            [
                'label' => esc_html__( 'Margin', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-description' => 'margin: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Padding. */
        $this->add_responsive_control(
            'description_padding',
            [
                'label' => esc_html__( 'Padding', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-description' => 'padding: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'style_color_description',
            [
                'label' => esc_html__( 'Ink Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-description' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'style_typography_description',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-pricer-description',
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'style_shadow_description',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-pricer-description',
            ]
        );

        /** Background */
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'description_background',
                'label' => esc_html__( 'Background', 'pricer-elementor' ),
                'types' => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .mdp-pricer-description',
            ]
        );

        /** End section style description. */
        $this->end_controls_section();

    }

    /**
     * Controls in the Style Tab – Button Section
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function style_button() {

        /** Style button tab. */
        $this->start_controls_section( 'style_button',
            [
                'label' => esc_html__( 'Button', 'pricer-elementor' ), 'tab' => Controls_Manager::TAB_STYLE,
            ] );

        /** Description width. */
        $this->add_responsive_control(
            'button__container_width',
            [
                'label' => esc_html__( 'Container Width', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'size_units' => [ '%', 'px', 'vh' ],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    'vh' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'desktop_default' => [
                    'size' => 100,
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        /** Margin. */
        $this->add_responsive_control(
            'button_margin',
            [
                'label' => esc_html__( 'Container Margin', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link' => 'margin: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Padding. */
        $this->add_responsive_control(
            'button_padding',
            [
                'label' => esc_html__( 'Container Padding', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link' => 'padding: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'toggle' => true,
            ]
        );

        /** Typography. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'style_typography_button',
                'label' => esc_html__( 'Typography', 'pricer-elementor' ),
                'scheme' => Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .mdp-pricer-link a',
            ]
        );

        /** Shadow. */
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'style_shadow_button',
                'label' => esc_html__( 'Shadow', 'pricer-elementor' ),
                'selector' => '{{WRAPPER}} .mdp-pricer-link a',
            ]
        );

        /** Button width. */
        $this->add_control(
            'button_custom_width',
            [
                'label' => esc_html__( 'Custom Width', 'pricer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'On', 'pricer-elementor' ),
                'label_off' => esc_html__( 'Off', 'pricer-elementor' ),
                'return_value' => 'yes',
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a' => 'display: inline-block;',
                ],
            ]
        );

        /** Custom width */
        $this->add_responsive_control(
            'button_width',
            [
                'label' => esc_html__( 'Button Width', 'pricer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ '%', 'px' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 300,
                    ],
                ],
                'default' => [
                    'unit' => '%',
                    'size' => 100,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [ 'button_custom_width' => 'yes' ]
            ]
        );

        $this->start_controls_tabs('tabs_link_style');

        $this->start_controls_tab(
            'tab_normal_link',
            [
                'label' => esc_html__('Normal', 'pricer-elementor')
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'style_color_button',
            [
                'label' => esc_html__( 'Ink color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Border padding. */
        $this->add_responsive_control(
            'button_border_padding',
            [
                'label' => esc_html__( 'Padding', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a' => 'padding: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
            ]
        );

        /** Border radius. */
        $this->add_responsive_control(
            'button_border_radius',
            [
                'label' => esc_html__( 'Border radius', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a' => 'border-radius: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ]
            ]
        );

        /** Background */
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'link_background',
                'label' => esc_html__( 'Background', 'pricer-elementor' ),
                'types' => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .mdp-pricer-link a',
            ]
        );

        /** Border type. */
        $this->add_control(
            'button_border_type',
            [
                'label' => esc_html__( 'Border', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'none',
                'options' => [
                    'none'  => esc_html__( 'None', 'pricer-elementor' ),
                    'double' => esc_html__( 'Double', 'pricer-elementor' ),
                    'dashed' => esc_html__( 'Dashed', 'pricer-elementor' ),
                    'dotted' => esc_html__( 'Dotted', 'pricer-elementor' ),
                    'solid' => esc_html__( 'Solid', 'pricer-elementor' ),
                    'grove' => esc_html__( 'Groove', 'pricer-elementor' ),
                ],
                'separator' => 'before',
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a' => 'border-style: {{VALUE}}',
                ],
            ]
        );

        /** Border color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'border_color_button',
            [
                'label' => esc_html__( 'Border color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_1,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a' => 'border-color: {{VALUE}}',
                ],
                'conditions' => [
                    'terms' => [
                        [
                            'name' => 'button_border_type',
                            'operator' => '!in',
                            'value' => [
                                'none',
                            ],
                        ],
                    ],
                ],
            ]
        );

        /** Border width. */
        $this->add_responsive_control(
            'button_border_width',
            [
                'label' => esc_html__( 'Border width', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a' => 'border-width: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'conditions' => [
                    'terms' => [
                        [
                            'name' => 'button_border_type',
                            'operator' => '!in',
                            'value' => [
                                'none',
                            ],
                        ],
                    ],
                ],
            ]
        );



        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_hover_link',
            [
                'label' => esc_html__('Hover', 'pricer-elementor')
            ]
        );

        /** Color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'style_hover_color_button',
            [
                'label' => esc_html__( 'Ink Color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        /** Border padding. */
        $this->add_responsive_control(
            'button_hover_border_padding',
            [
                'label' => esc_html__( 'Padding', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a:hover' => 'padding: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
            ]
        );

        /** Border radius. */
        $this->add_responsive_control(
            'button_hover_border_radius',
            [
                'label' => esc_html__( 'Border radius', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a:hover' => 'border-radius: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
            ]
        );

        /** Background */
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'hover_link_background',
                'label' => esc_html__( 'Background', 'pricer-elementor' ),
                'types' => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .mdp-pricer-link a:hover',
            ]
        );

        /** Hover Animation. */
        $this->add_control(
            'hover_link_animation',
            [
                'label' => esc_html__( 'Hover Animation', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'none' => 'None',
                    'hvr-grow' => 'Grow',
                    'hvr-shrink' => 'Shrink',
                    'hvr-pulse' => 'Pulse',
                    'hvr-pulse-grow' => 'Pulse Grow',
                    'hvr-pulse-shrink' => 'Pulse Shrink',
                    'hvr-push' => 'Push',
                    'hvr-pop' => 'Pop',
                    'hvr-bounce-in' => 'Bounce In',
                    'hvr-bounce-out' => 'Bounce Out',
                    'hvr-rotate' => 'Rotate',
                    'hvr-grow-rotate' => 'Grow Rotate',
                    'hvr-float' => 'Float',
                    'hvr-sink' => 'Sink',
                    'hvr-bob' => 'Bob',
                    'hvr-hang' => 'Hang',
                    'hvr-skew' => 'Skew',
                    'hvr-skew-forward' => 'Skew Forward',
                    'hvr-skew-backward' => 'Skew Backward',
                    'hvr-wobble-horizontal' => 'Wobble Horizontal',
                    'hvr-wobble-vertical' => 'Wobble Vertical',
                    'hvr-wobble-to-bottom-right' => 'Wobble To Bottom Right',
                    'hvr-wobble-to-top-right' => 'Wobble To Top Right',
                    'hvr-wobble-top' => 'Wobble Top',
                    'hvr-wobble-bottom' => 'Wobble Bottom',
                    'hvr-wobble-skew' => 'Wobble Skew',
                    'hvr-buzz' => 'Buzz',
                    'hvr-buzz-out' => 'Buzz Out',
                    'hvr-forward' => 'Forward',
                    'hvr-backward' => 'Backward',
                ],
                'default' => 'none',
            ]
        );

        /** Border type. */
        $this->add_control(
            'button_hover_border_type',
            [
                'label' => esc_html__( 'Border', 'pricer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'none',
                'options' => [
                    'none'  => esc_html__( 'None', 'pricer-elementor' ),
                    'double' => esc_html__( 'Double', 'pricer-elementor' ),
                    'dashed' => esc_html__( 'Dashed', 'pricer-elementor' ),
                    'dotted' => esc_html__( 'Dotted', 'pricer-elementor' ),
                    'solid' => esc_html__( 'Solid', 'pricer-elementor' ),
                    'grove' => esc_html__( 'Groove', 'pricer-elementor' ),
                ],
                'separator' => 'before',
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a:hover' => 'border-style: {{VALUE}}',
                ],
            ]
        );

        /** Border color. */
        /** @noinspection PhpUndefinedClassInspection */
        $this->add_control(
            'border_hover_color_button',
            [
                'label' => esc_html__( 'Border color', 'pricer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' =>  Color::get_type(),
                    'value' =>  Color::COLOR_4,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a:hover' => 'border-color: {{VALUE}}',
                ],
                'conditions' => [
                    'terms' => [
                        [
                            'name' => 'button_hover_border_type',
                            'operator' => '!in',
                            'value' => [
                                'none',
                            ],
                        ],
                    ],
                ],
            ]
        );

        /** Border width. */
        $this->add_responsive_control(
            'button_hover_border_width',
            [
                'label' => esc_html__( 'Border width', 'pricer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-pricer-link a:hover' => 'border-width: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
                ],
                'conditions' => [
                    'terms' => [
                        [
                            'name' => 'button_hover_border_type',
                            'operator' => '!in',
                            'value' => [
                                'none',
                            ],
                        ],
                    ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        /** End section style button. */
        $this->end_controls_section();

    }

    /**
     * Add the widget controls.
     *
     * @since 1.0.0
     * @access protected
     *
     * @return void with category names
     **/
    protected function register_controls() {

        /** Content – Layout */
        $this->content_layout();

        /** Content – Header */
        $this->content_header();

        /** Content – Price */
        $this->content_price();

        /** Content – Period */
        $this->content_period();

        /** Content – Features */
        $this->content_features();

        /** Content – Description */
        $this->content_description();

        /** Content – Button */
        $this->content_button();

        /** Style – Header */
        $this->style_header();

        /** Style – Subheader */
        $this->style_subheader();

        /** Style – Price */
        $this->style_price();

        /** Style – Period */
        $this->style_period();

        /** Style – Divider */
        $this->style_divider();

        /** Style – Features */
        $this->style_features();

        /** Style – Description */
        $this->style_description();

        /** Style – Button */
        $this->style_button();

    }

    /**
     * We return an array with the names of the layouts and their width.
     *
     * @param $settings - we get an array with all the data.
     *
     * @return array
     */
    public function get_LayoutSortable( $settings ) {

        /** We get all the data on the lists of layouts. */
        $layout = $settings['plan_layout'];

        /** An array in which the names of the layouts and their width will be stored. */
        $res = array();

        /** We write the names of the layouts and their width into an array. */
        foreach ($layout as $key => $val){
            $res[] = [ 'name' => $val['plan_layout_select'], 'width' => isset($val['box_width']['size']) ];
        }

        return $res;

    }

    /**
     * We display the entire list of proposals for the tariff plan.
     *
     * @param $settings - Array with data on the tariff plan.
     * @param $iconActive - The class of the active icon.
     * @param $iconInActive - The class of the inactive icon.
     *
     * @return void
     */
    public function get_tariffPlan( $settings, $iconActive, $iconInActive ) {

        echo '<ul class="mdp-pricer-list">';

        foreach ( $settings as $item ) {
            if ($item['plan_included'] === 'yes') {
                echo '<li class="mdp-pricer-active-features"><i class="' . $iconActive . ' mdp-pricer-active-icon-color"></i>' . $item['description_text'] . '</li>';
            }else{
                echo '<li class="mdp-pricer-inactive-features"><i class="' . $iconInActive . ' mdp-pricer-inactive-icon-color"></i>' . $item['description_text'] . '</li>';
            }
        }

        echo '</ul>';

    }

    /**
     * Render Frontend Output. Generate the final HTML on the frontend.
     *
     * @since 1.0.0
     * @access protected
     **/
    protected function render() {

        /** We get all the values from the admin panel. */
        $settings = $this->get_settings_for_display();

        /** The class of the active icon. */
        $iconActive = $settings['icon_active']['value'];

        /** The class of the inactive icon. */
        $iconInActive = $settings['icon_inactive']['value'];

        /** We return an array with the names of the layouts and their width. */
        $LayoutSortable = $this->get_LayoutSortable($settings);

        /** We wrap all the contents in the block if the orientation of the tariff plan is horizontal. */
        echo '<div class="mdp-pricer">';

        $plan_array = null;

        /** We list the layouts in the given order. */
        foreach ($LayoutSortable as $val){

            if( isset($settings['plan_'.$val['name']]) ){
                $plan_array = $settings['plan_'.$val['name']];
            }

            /** If this is a list with tariff plans, then we display it. */
            if( is_array( $plan_array ) && $val['name'] === 'list'){

                /** We display the conditions of the tariff plan. */
                $this->get_tariffPlan($settings[ 'plan_' . $val['name']], $iconActive, $iconInActive );

                /** If this is a header, then we display it. */
            }elseif( $val['name'] === 'header' ){

                /** We determine whether to make the background of the block the background of the header. */
                $settings['header_background_effect'] === 'yes' ? $classHEffect = 'mdp-pricer-header-effect' : $classHEffect = '';

                /** We determine whether to make the background of the block the background of the subheader. */
                $settings['subheader_background_effect'] === 'yes' ? $classSHEffect = 'mdp-pricer-subheader-effect' : $classSHEffect = '';

                echo '<' . esc_attr( $settings['plan_tag'] ) . ' class="mdp-pricer-heading" >';

                /** Display the subheader above the title. */
                if( $settings['show_sub_title'] === 'yes' and $settings['sub_title_position'] === 'top' ){
                    echo '<span class="mdp-pricer-subheader '. esc_attr( $classSHEffect ) . '">' . esc_html( $settings['plan_sub_name'] ). '</span>';
                }

                /** Display the header. */
                echo '<span class="mdp-pricer-header ' . esc_attr( $classHEffect ) . '">' . esc_html( $settings['plan_'.$val['name']] ) . '</span>';

                /** Display the subheader. */
                if($settings['show_sub_title'] === 'yes' and $settings['sub_title_position'] === 'bottom' ){
                    echo '<span class="mdp-pricer-subheader ' . esc_attr( $classSHEffect ) . '">' . esc_html( $settings['plan_sub_name'] ) . '</span>';
                }

                echo '</' . esc_attr( $settings['plan_tag'] ) . '>';

                /** If this is a price, then we display it. */
            }elseif( $val['name'] === 'price' ){

                /** Received currency location. */
                $currency = explode("-", $settings['currency_position'] );

                /** We break the price before and after the floating point. */
                $price = explode(".", $settings['plan_'.$val['name']]);

                /** We display a custom separator for the floating point price. */
                if( $settings['decimal_position'] === 'default' ){
                    if ( count($price) > 1 ) {
                        $point = $settings['decimal_point'];
                    } elseif ( count( $price ) === 1 && $settings['show_zeros_original_price'] === 'yes' ) {
                        $point = $settings['decimal_point'];
                        $price[1] = '00';
                    } else {
                        $point = '';
                        $price[1] = '';
                    }
                } else {
                    $point = '';
                }

                /** Show / Hide tenths */
                if( is_int($settings['plan_original_price']) and $settings['show_zeros_original_price'] !== 'yes' ){
                    $decimals = 0;
                } else {
                    $decimals = 2;
                }

                echo '<p class="mdp-pricer-price">';

                if ($settings['original_price_position'] === 'left') {

                    /** We display the currency symbol on the left side. */
                    if ( $currency[0] === 'left' ) {
                        echo '<span class="mdp-original-pricer-currency">' . esc_html( $settings['plan_currency'] ) . '</span>';
                    }

                    echo '<span class="mdp-original-pricer-style">' . number_format( $settings['plan_original_price'], $decimals, $settings['decimal_point'], ' ' ) . '</span>';

                    /** We display the currency symbol on the right side. */
                    if ( $currency[0] === 'right' ) {
                        echo '<span class="mdp-original-pricer-currency">' . esc_html( $settings['plan_currency'] ) . '</span>';
                    }

                }

                /** We display the currency symbol on the left side. */
                if ( $currency[0] === 'left' ) {
                    echo '<span class="mdp-pricer-currency mdp-pricer-currency-position-'.$settings['currency_position'].'">' . $settings['plan_currency'] . '</span>';
                }

                /** Print the value of the floating point. */
                echo '<span class="mdp-pricer-style">' . number_format( $price[0] ) .'<span class="mdp-pricer-decimal mdp-pricer-decimal-position-' . esc_attr( $settings['decimal_position'] ) .'">' . esc_html( $point ) . esc_html( $price[1] ) . '</span></span>';

                /** We display the currency symbol on the right side. */
                if ( $currency[0] === 'right' ) {
                    echo '<span class="mdp-pricer-currency mdp-pricer-currency-position-' . esc_attr( $settings['currency_position'] ) .'">' . esc_html( $settings['plan_currency'] ) . '</span>';
                }

                if ($settings['original_price_position'] === 'right') {

                    /** We display the currency symbol on the left side. */
                    if ( $currency[0] === 'left' ) {
                        echo '<span class="mdp-original-pricer-currency">' . esc_html( $settings['plan_currency'] ) . '</span>';
                    }

                    /** Display price. */
                    echo '<span class="mdp-original-pricer-style">' . number_format( $settings['plan_original_price'], $decimals, $settings['decimal_point'], ' ' ) . '</span>';

                    /** We display the currency symbol on the right side. */
                    if ( $currency[0] === 'right' ) {
                        echo '<span class="mdp-original-pricer-currency">' . esc_html( $settings['plan_currency'] ) . '</span>';
                    }

                }

                echo '</p>';

                /** If this is a period, then we display it. */
            }elseif( 'period' === $val[ 'name' ] ){

                echo '<' . esc_attr( $settings['period_tag'] ) . ' class="mdp-pricer-period">' . esc_html( $settings[ 'plan_'.$val['name'] ] ) . '</' . esc_attr( $settings['period_tag'] ) . '>';

                /** If this is a description, then we display it. */
            }elseif( 'description' === $val[ 'name' ] ){

                echo '<' . esc_attr( $settings['description_tag'] ) . ' class="mdp-pricer-description">' . esc_html( $settings['plan_'.$val['name']] ) . '</' . esc_attr( $settings['description_tag'] ) . '>';

                /** If this is a divider, then we display it. */
            }elseif( 'divider' === $val[ 'name' ] ){

                echo '<div class="mdp-pricer-box-divider"><hr class="mdp-pricer-divider"></div>';

                /** If this is a link, then we display it. */
            } elseif (is_array($settings['plan_'.$val['name']]) && $val['name'] === 'link') {

                $target = isset($settings['plan_link']['is_external']) ? ' target="_blank"' : '';
                $nofollow = isset($settings['plan_link']['nofollow']) ? ' rel="nofollow"' : '';
                echo '<p class="mdp-pricer-link"><a href="' . esc_url( $settings['plan_link']['url'] ) . '" class="' . esc_attr( $settings['hover_link_animation'] ) . '" ' . $target . $nofollow . ' > ' . esc_html( $settings['link_text'] )  . ' </a></p>';
                /** If this is another layout list, then we display it. */
            } else {

                echo 'Unknown layout';

            }

        }

        echo '</div>';

    }

    /**
     * Return link for documentation.
     *
     * Used to add stuff after widget.
     *
     * @since 1.0.0
     * @access public
     **/
    public function get_custom_help_url() {
        return 'https://docs.merkulov.design/category/pricer-elementor/';
    }

}
